/* This magma code accompanies the article B. Banwait, F. Fité, D. Loughran - Del Pezzo surfaces over finite fields and their Frobenius traces.

Its purpose to verify that Tables 2 and 3 on p178 of Manin's book concerning minimal quartic del Pezzo surfaces over arbitrary fields are correct

*/


/* Returns true if the given orbit consists of skew curves. */
is_skew := function(O, M)
 for i in O do
  for j in O do
   if i ne j and M[i,j] ne 0 then
    return false;
   end if;
  end for;
 end for;
 return true;
end function;

/*The next function returns true if the given configurations are equivalent, in the sense that they differ by a translation */
IsEquivalent := function(O1,O2, we)
 for w in we do
  if O1^w eq O2 then
   return true;
  end if;
 end for;
 return false;
end function;

/*Prints the graph of lines of the given configuration */
Print_Graph := procedure(O)
 O:=SetToSequence(O);
 vertices:= [12,10,6,16,4,2,14,8,3,5,9,15,11,13,1,7];
 for v in [1..#vertices] do
  for o in O do
   if vertices[v] in o then
    printf "%o ", Position(O,o);
    break o;
   end if;
  end for;
  if v mod 4 eq 0 then
   printf "\n";
  end if;
 end for;
 printf "\n";
end procedure;

/*This code calculates the first Galois cohomology of the Picard group using [Manin, Proposition 31.3]. It was kindly given to us by Jörg Jahnel */
H1Pic := function(gr, deg, M)
 /* Jede Brauergruppe wird hier eingetragen werden. */

 /* Die Divisoren ueber dem algebraischen Abschlusz, zum Beispiel

 Z^27 bei kubischen Flaechen. */
 Div_space := RSpace(Integers(), deg);
 Hom_alg := Hom(Div_space, Div_space);
 int_map := Hom_alg!M;
 /* Die Hauptdivisoren */
 S_0 := Kernel(int_map);


 /* Alles haengt nur von der kombinatorischen Struktur der
 Orbiten ab. */
 orb := Orbits(gr);

 /* Die invarianten Divisoren */
 Div_space_arith := RSpace(Integers(), #orb);

 /* Die Normabbildung */
 N := ZeroMatrix(Integers(), deg,#orb);  
 for j := 1 to #orb do
  akt := orb[j];
  for k := 1 to #akt do
   /* Die Gerade akt[k] liegt im Orbit akt mit der Nummer j. */
   N[akt[k]][j] := #gr/#akt;
  end for;
 end for;
 norm_map := Hom(Div_space, Div_space_arith)!N;
 NS := Image(norm_map);

 /* Das Bild der Hauptdivisoren unter N. */
 NS_0 := norm_map(S_0);

 /* Die Einbettung der ionvarianten Divisoren in den Modul aller
 Divisoren */
 O := ZeroMatrix(Integers(), #orb,deg);
 for j := 1 to #orb do
  akt := orb[j];
  for k := 1 to #akt do
   /* Die Gerade akt[k] liegt im Orbit akt mit der Nummer j. */
   O[j][akt[k]] := 1;
  end for;
 end for;
 orb_map := Hom(Div_space_arith, Div_space)!O;
 
 /* int_map \circ orb_map !!! */
 /* NS geschnitten mit S_0 */
 linke_seite := NS meet Kernel(orb_map * int_map);

 /* Die Brauergruppe nach [Manin, Proposition 31.3] */
 Br := linke_seite / NS_0;
 vec := Moduli(Br);  /* Die abelschen Invarianten */
 return vec;
end function;

/* Runs through the given list of subgroups and determines how many
are minimal */
properties_of_the_subgr_classes := function(list, deg,M)
 erg := [];
 n_min := 0;
 for rec in list do
  gr := rec`subgroup;

  minimal:=true;
  for O in Orbits(gr) do
   if is_skew(O,M) then
    minimal:=false;
    break O;
   end if;
  end for;

  if minimal then
   Append(~erg, gr);
   n_min := n_min + 1;
  end if;
 end for;
 printf "%o are minimal,\n", n_min;
 return erg;
end function;


/************************** Degree 4 *******************************
*                                                                  *
*******************************************************************/
/* Search for subgroups of W(D_5) whose corresponding surface is minimal. */
printf "W(D_5)\n";
wd5 := TransitiveGroup(16, 1328);
R_d5 := RootDatum("D5");
Cox_d5 := CoxeterGroup(R_d5);
/* IsIsomorphic(wd5, Cox_d5); */

/* Pairs of numbers in {1,..,16}. */
subs2 := Subsets(Set(GSet(wd5)), 2);
orb := Orbits(wd5, GSet(wd5, subs2));
/* #orb[1]; */
/* 40 */
/* #orb[2]; */
/* 80 */

/* The intersection matrix of the Clebsch graph*/
M := ZeroMatrix(Integers(), 16,16);
for i := 1 to 16 do
 for j := 1 to 16 do
  if i eq j then M[i,j] := -1; end if;
  if {i,j} in orb[1] then M[i,j] := 1; end if;
 end for;
end for;

list_wd5 := SubgroupClasses(wd5);
printf "%o conjugacy classes of subgroups in total,\n", #list_wd5;
erg := properties_of_the_subgr_classes(list_wd5, 16,M);

/*The list of possible decompositions claimed by Manin */
Manin:=
[{ {11,13,1,7}, {3,5,9,15,4,2,14,8,12,10,6,16} },
{ {11,13}, {1,7}, {3,5,4,2,12,10}, {9,15,14,8,6,16} },
{ {11,13}, {3,5,4,2,12,10}, {1,7,9,15,14,8,6,16} },
{ {11,13}, {3,5}, {4,2}, {12,10}, {1,7}, {9,15}, {14,8}, {6,16} },
{ {11,13,2,4}, {3,5,10,12}, {1,7,9,15}, {14,8,6,16} },
{ {11,13}, {3,5}, {4,2}, {12,10}, {1,7,9,15}, {14,8,6,16} },
{ {11,13}, {3,5}, {4,2,12,10}, {1,7}, {9,15}, {14,8,6,16} },
{ {11,13}, {3,5}, {4,2,12,10}, {1,7,9,15}, {14,8,6,16} },
{ {11,13,3,5}, {4,2,12,10}, {1,7,9,15}, {14,8,6,16} },
{ {11,13,1,7}, {3,5,9,15}, {4,2,14,8}, {12,10,6,16} },
{ {11,13,1,7}, {3,2,9,8}, {4,5,14,15}, {12,10,6,16} },
{ {11,13}, {3,5}, {4,2}, {12,10}, {1,7,9,15,14,8,6,16} },
{ {11,13}, {3,5}, {4,2,12,10}, {1,7,9,15,14,8,6,16} },
{ {11,13,3,5}, {4,2,12,10}, {1,7,9,15,14,8,6,16} },
{ {11,13,3,5,4,2,12,10}, {1,7,9,15,14,8,6,16} },
{ {11,13,1,7}, {3,5,9,15}, {4,2,12,10,14,8,6,16} },
{ {11,13,1,7,3,5,9,15}, {4,2,12,10,14,8,6,16} },
{ {11,13,1,7,3,9,2,8}, {5,15,4,14,12,10,6,16} },
{ {11,13,1,7,3,5,9,15,4,2,12,10,14,8,6,16} } ];

/* Print Manin's list to confirm that the graphs agree with what he claims*/
print " ";
print "Graphs of Manin's configurations (the vertices numbered according to which orbit they belong):";
for m in Manin do
 print Position(Manin,m);
 Print_Graph(m);
end for;

/* First we check that all possible cases listed by Manin actually occur*/
print "Check that all cases listed by Manin actually occur, and that their H^1(Pic) is as claimed:";
for i in [1..#Manin] do
 for e in erg do
  BOrbit:={};
  for o in Orbits(e) do
   BOrbit:=Include(BOrbit,Set(o));
  end for;

  if IsEquivalent(BOrbit,Manin[i],wd5) then
   printf "%o. H^1(Pic) = %o \n",i, H1Pic(e,16,M);
   break e;
  end if;
 end for;
end for;

/* Next we confirm that none of Manin's cases are equivalent */
print " ";
print "Check that none of Manin's cases are equivalent (1 2 false means that configuration 1 is not equivalent to configuration 2):";
for i in [1..#Manin] do
 for j in [i..#Manin] do
  print i,j, IsEquivalent(Manin[i], Manin[j], wd5);
 end for;
end for;

/*
IsEquivalent({ {1,11}, {2,8}, {3,9}, {4,14}, {5,15}, {6,12}, {7,13}, {10,16} }, { {1,2}, {3,16}, {4,7}, {5,6}, {8,11}, {9,10}, {12,15}, {13,14} }, wd5);
/*



/* Finally, confirm that Manin did not miss any cases */
print " ";
print "Confirm that Manin did not miss any cases (takes about 10 seconds):";
New_case:=false;
for e in erg do
 BOrbit:={};
 for o in Orbits(e) do
  BOrbit:=Include(BOrbit,Set(o));
 end for;

 is_new := true;
 for m in Manin do
  if IsEquivalent(BOrbit, m, wd5) then
   is_new:=false;
  end if;
 end for;

 if is_new then
  print e, BOrbit;
  New_case:=true;
 end if;
end for;

if New_case eq false then
 print "No new cases found!";
end if;

