/* This magma code accompanies the article B. Banwait, F. Fité, D. Loughran - Del Pezzo surfaces over finite fields and their Frobenius traces.

Its purpose is to search for collections of closed points in P^2 over a finite field F_q in general position. If they exist it finds them (usually very quickly), and if they don't exist it proves it. Proving certain collections don't exist can be quite time consuming for q > 10. The criterion for determining whether points lie in general position is Lemma 2.5 from the paper.

See the end of the code for some example applications.

If you run the programme straight off, then it verifies the claims made in the paper concerning existence and non-existence of various collections of points in general position. This takes a couple of hours. To play around with it by yourself, set "Check_claims" below to be false.

*/

Check_claims:=true;

/* Given a set of points P over a field k, outputs true iff no three are colinear. */
NoThreeCollinear := function(P, k)
 for Three in Subsets(P,3) do
  T:=SetToSequence(Three);
  M:=Matrix(k,3,3,T);
  if Determinant(M) eq 0 then
   return false;
  end if;
 end for;
 return true;
end function;

/* Given a set of points P over a field k, outputs true iff no six lie on a conic */
NoSixCoconic := function(P, k)
 for Six in Subsets(P,6) do
  T:=SetToSequence(Six);
  R:=[];
  for t in T do
   Rt:=[t[1]^2,t[2]^2,t[3]^2,t[1]*t[2],t[1]*t[3],t[2]*t[3]];
   R:=Append(R,Rt);   
  end for;
  M:=Matrix(k,6,6,R);
  if Determinant(M) eq 0 then 
   return false;
  end if;
 end for;
 return true;
end function;

/* Given a set of points P over a field k, outputs true iff no eight lie on a cubic with a singularity at one of the points */
NoEightCosingularcubic := function(P, k)
 P2<x,y,z>:=ProjectiveSpace(k,2);
 E:=Scheme(P2, x^3 + z^3 - y^2*z);
 for Eight in Subsets(P,8) do
  T:=SetToSequence(Eight);
  R:=[];
  for t in T do
   Rt:=[ t[1]^3, t[2]^3, t[3]^3, t[1]^2*t[2], t[1]^2*t[3], t[1]*t[2]^2, t[2]^2*t[3], t[1]*t[3]^2, t[2]*t[3]^2, t[1]*t[2]*t[3] ];
   R:=Append(R,Rt);   
  end for;
  for t in T do
   dx:=[3*t[1]^2,0,0,2*t[1]*t[2],2*t[1]*t[3],t[2]^2,0,t[3]^2,0,t[2]*t[3]];
   dy:=[0,3*t[2]^2,0,t[1]^2,0,2*t[1]*t[2],2*t[2]*t[3],0,t[3]^2,t[1]*t[3]];
   dz:=[0,0,3*t[3]^2,0,t[1]^2,0,t[2]^2,2*t[1]*t[3],2*t[2]*t[3],t[1]*t[2]];
   M:=Matrix(k,11,10, R cat [dx, dy, dz]);
   if Rank(M) ne 10 then
    return false;
   end if;
  end for;
 end for;
 return true;
end function;

/*Combines the previous three functions to check whether a collection of points P over a field k lies in general position */
InGeneralPos := function(P, k)
 if NoThreeCollinear(P,k) and NoSixCoconic(P,k) and NoEightCosingularcubic(P,k) then
  return true;
 else
  return false;
 end if;
end function;

/* Given a selection of points PStart over a finite field k, outputs all rational points which lie in general position which respect to PStart */
RemainingPointsInGenPos := function(PStart,k)
 r:=#PStart;
 Out:={};
 NonZero:=Exclude(Set(k),0);
 for x,y in NonZero do
  P:=PStart join {[x,y,1]};
  if #P eq r+1 and InGeneralPos(P,k) then
    Out:=Out join {[x,y,1]};
  end if;
 end for;
 return Out;
end function;

/* Input: a prime power q, a "type" T, and a boolean variable display. Returns true iff there are a collection of closed points of Type T in P^2 over F_q in general position. If display = true, then it also outputs an example of such a collection of points, if it exists.

Examples: T=[7] - 7 rational points in general position. T=[1,1,1] - 1 rational point, 1 closed point degree 2, 1 closed point degree 3. T=[4,2] - 4 rational points and 2 closed points of degree 2. T = [0,0,0,0,0,1] - 1 closed point degree 6 */
HasPointsInGenPos := function(q,T,display)
 r:=0; //r is the number of points over kbar corresponding to the type T
 for t in [1..#T] do
  r := r + t*T[t];
 end for;

 max:=1;   /*max = largest d for which a field field of order q^d occurs*/
 embed:=1; /*embed = largest d for which all relevant fields embed into a finite field of order q^d */
 for t in [1..#T] do
  if T[t] ne 0 then
   max:=t;
   embed:=LCM(embed,t);
  end if;
 end for;

 PStart:={}; /*If some of the points are rational, then WLOG we may assume that they are [1,0,0], [0,1,0], etc... */
 if T[1] gt 0 then PStart:=PStart join { [1,0,0] }; end if;
 if T[1] gt 1 then PStart:=PStart join { [0,1,0] }; end if;
 if T[1] gt 2 then PStart:=PStart join { [0,0,1] }; end if;
 if T[1] gt 3 then PStart:=PStart join { [1,1,1] }; end if;
 
 /*Set-up the relevant finite fields which will be needed*/
 k1<u1>:=FiniteField(q);
 k2<u2>:=FiniteField(q^2);
 k3<u3>:=FiniteField(q^3);
 k4<u4>:=FiniteField(q^4);
 k5<u5>:=FiniteField(q^5);
 k6<u6>:=FiniteField(q^6);
 k7<u7>:=FiniteField(q^7);
 k8<u8>:=FiniteField(q^8);
 L:=FiniteField(q^embed);
 Fields:=[k1,k2,k3,k4,k5,k6,k7,k8];

 /*Create the set which we shall loop over. We loop over points defined over a finite extension and then force them to be Galois invariant by taking conjugates.
 The first part is ad hoc and just makes sure Looping has the right universe*/
 Looping:=[Set(L)];
 Looping:=Exclude(Looping, Set(L));

 for t in [1..#T] do
  if t eq 1 and T[1] gt 4 then
   for tt in [1..T[1] - 4] do
    Looping:=Append(Looping, Set(Fields[1]) diff Set([0])); 
   end for;
  end if;
  if t ne 1 then
   for tt in [1..T[t]] do
    Looping:=Append(Looping, Set(Fields[t]) diff Set(Fields[t-1]));
   end for;
  end if;
 end for;

 /*First look at random collections of points to try to find some in general position. This speeds the algorithm up a lot.*/
 for i in [1..10000] do
  x:=Random(CartesianProduct(Looping));
  y:=Random(CartesianProduct(Looping));
  P:=PStart;
  for t in [1..#x] do
   for j in [0..max-1] do
    P:=P join { [x[t]^(q^j), y[t]^(q^j), 1] };  /*Take the conjugates of the given point to force a Galois invariant set*/
   end for;	
  end for;
  if #P eq r and InGeneralPos(P,L) then  /*Found points in general position*/
   if display then
    print q, T, P;
    return true;
   end if;
   return q, T, true;
  end if;
 end for;

 /*If no random collections of points in general position found, then there are probably none in general position. So try to rigorously prove this. This step can take quite some time for large q*/
 for x,y in CartesianProduct(Looping) do
  P:=PStart;
  for t in [1..#x] do
   for j in [0..max-1] do
    P:=P join { [x[t]^(q^j), y[t]^(q^j), 1] };  /*Take the conjugates of the given point as required*/
   end for;	
  end for;
  if #P eq r and InGeneralPos(P,L) then
   if display then
    print q, T, P;
    return true;
   end if;
   return q, T, true;
  end if;
 end for;
 
 /*No points in general position*/
 return q, T, false; 
end function;

/*This function is very similar to the previous function "HasPointsInGenPos". The sole difference is that it only searches randomly for points in general position, it does
not prove that they don't exist. It returns false if it does not find any; in practice this means that they don't exist, as points in general position are usually found very quickly when they exist */
SearchPointsInGenPos := function(q,T,display)
 r:=0; //r is the number of points over kbar corresponding to the type T
 for t in [1..#T] do
  r := r + t*T[t];
 end for;

 max:=1;   /*max = largest d for which a field field of order q^d occurs*/
 embed:=1; /*embed = largest d for which all relevant fields embed into a finite field of order q^d */
 for t in [1..#T] do
  if T[t] ne 0 then
   max:=t;
   embed:=LCM(embed,t);
  end if;
 end for;

 PStart:={}; /*If some of the points are rational, then WLOG we may assume that they are [1,0,0], [0,1,0], etc... */
 if T[1] gt 0 then PStart:=PStart join { [1,0,0] }; end if;
 if T[1] gt 1 then PStart:=PStart join { [0,1,0] }; end if;
 if T[1] gt 2 then PStart:=PStart join { [0,0,1] }; end if;
 if T[1] gt 3 then PStart:=PStart join { [1,1,1] }; end if;
 
 /*Set-up the relevant finite fields*/
 k1<u1>:=FiniteField(q);
 k2<u2>:=FiniteField(q^2);
 k3<u3>:=FiniteField(q^3);
 k4<u4>:=FiniteField(q^4);
 k5<u5>:=FiniteField(q^5);
 k6<u6>:=FiniteField(q^6);
 k7<u7>:=FiniteField(q^7);
 k8<u8>:=FiniteField(q^8);
 L:=FiniteField(q^embed);
 Fields:=[k1,k2,k3,k4,k5,k6,k7,k8];

 /*Create the set which we shall loop over. We loop over points defined over a finite extension and then force them to be Galois invariant by taking conjugates.
 The first part is ad hoc and just makes sure Looping has the right universe*/
 Looping:=[Set(L)];
 Looping:=Exclude(Looping, Set(L));

 for t in [1..#T] do
  if t eq 1 and T[1] gt 4 then
   for tt in [1..T[1] - 4] do
    Looping:=Append(Looping, Set(Fields[1]) diff Set([0])); 
   end for;
  end if;
  if t ne 1 then
   for tt in [1..T[t]] do
    Looping:=Append(Looping, Set(Fields[t]) diff Set(Fields[t-1]));
   end for;
  end if;
 end for;

 /*First look at random collections of points to try to find some in general position. This speeds the algorithm up a lot.*/
 for i in [1..10000] do
  x:=Random(CartesianProduct(Looping));
  y:=Random(CartesianProduct(Looping));
  P:=PStart;
  for t in [1..#x] do
   for j in [0..max-1] do
    P:=P join { [x[t]^(q^j), y[t]^(q^j), 1] };  /*Take the conjugates of the given point as required*/
   end for;	
  end for;
  if #P eq r and InGeneralPos(P,L) then
   if display then
    print q, T, P;
    return true;
   end if;
   return q, T, true;
  end if;
 end for;
 
 /*No points in general position found after randomly searching*/
 return q, T, false; 
end function;





/* The programme now verifies the claims in the paper regarding existence and non-existence of various points in general position. This takes a couple of hours. To play around with it by yourself, set "Check_claims" at the start to be false. */

if Check_claims then
 display:=false;

 print " ";
 print "We now verify the claims in the paper regarding existence of collections of closed points in general position.";
 print "The output is: q, Type of configuration, Does this required configuration exist?";
 print "Examples of types: [7] = 7 rational points in general position. [4,0,1] = 4 rational points and a closed point of degree 3";
 print " ";

 /*First consider del Pezzo surfaces of degree 2*/
 print "Del Pezzo surfaces of degree 2.";
 for T in [[7], [5,1], [4,0,1]] do
  for q in [2..16] do 
   if IsPrimePower(q) then
    HasPointsInGenPos(q,T,display);  
   end if;
  end for;
  print " ";
 end for;

 /*Now consider del Pezzo surfaces of degree 1*/
 print "Del Pezzo surfaces of degree 1.";

 /*First consider 8 rational points in general position.
 Sometimes it does not find 8 rational points in general position when q=16 after randomly searching. If it does not immediately output "16 [ 8 ] true", then re-run the programme.
 Don't try to use this code to verify that there are *not* 8 rational points in general position when q=17! You will be waiting a very long time....*/
 print "Sometimes it does not find 8 rational points in general position when q=16 after randomly searching. If it does not immediately output 16 [ 8 ] true, then re-run the programme.";
 for q in [16] cat [19..49] do 
  if IsPrimePower(q) then
   HasPointsInGenPos(q,[8],display); /*8 rational points. a = 9*/
 end if;
 end for;
 print " ";

 print "We now look for 6 rational points and a closed point of degree 2 in general position. This takes a couple of hours.";
 for q in [2..29] do 
  if IsPrimePower(q) then
   HasPointsInGenPos(q,[6,1],display); /*6 rational points and a closed point of degree 2. a = 7. It takes around an hour and a half to check non-existence here for q=9*/
  end if;
 end for;
 print " ";

 for q in [2..17] do 
  if IsPrimePower(q) then
   HasPointsInGenPos(q,[5,0,1],display);  /*5 rational points and a closed point of degree 3. a = 6*/
  end if;
 end for;
 print " ";

 for T in [[5,0,1],[4,0,0,1],[3,0,0,0,1],[2,0,0,0,0,1]] do
  for q in [2..13] do 
   if IsPrimePower(q) then
    HasPointsInGenPos(q,T,display);
   end if;
  end for;
  print " ";
 end for;
end if;


/*Some sample points with interesting properties*/
//P:={ [1,0,0], [0,1,0], [0,0,1], [1,1,1], [2,3,1], [3,4,1] }; /*q=5: All three collinear, but all six lie on the conic x*y - 2*x*z + y*z */
//P:={ [1,0,0], [0,1,0], [0,0,1], [1,1,1], [1,1,2], [3,4,1] }; /*q=5: No three collinear, but not all six lie on a conic */

/*Some examples of 5 points in general position*/
//P:={ [1,0,0], [0,1,0], [0,0,1], [1,1,1], [2,6,1] }; /*q=7*/
/*P:={ [1,0,0], [0,1,0], [0,0,1], [1,1,1], [a,a^2,1] }; q=8*/

/*The famous collection of 6 rational points in general position when q=4
k<u>:=FiniteField(4);
P:={ [1,0,0], [0,1,0], [0,0,1], [1,1,1], [u,u^2,1] };
InGeneralPos(P,k);*/


/*These points lie in general position over the rationals. Let's see what happens when we reduce them modulo some primes 
P:={ [1,0,0], [0,1,0], [0,0,1], [1,1,1], [2,3,1], [4,5,1], [5,4,1], [6,2,1] };
k:=Rationals();
print P, NoThreeCollinear(P,k), NoSixCoconic(P,k), NoEightCosingularcubic(P,k);
for q in [2..30] do
 if IsPrime(q) then 
  k<u>:=FiniteField(q);
  print q, NoThreeCollinear(P,k), NoSixCoconic(P,k), NoEightCosingularcubic(P,k);
 end if;
end for;*/


/*These points lie on the cuspidal cubic curve over F_7
P:= { [0,0,1], [0,1,0], [1,1,1], [1,6,1], [2,1,1], [2,6,1], [4,1,1], [4,6,1] };
k:=FiniteField(7);
NoThreeCollinear(P,k);
NoSixCoconic(P,k);
NoEightCosingularcubic(P,k); */

/*These points lie on the nodal cubic curve over F_11
P:= { [0,0,1], [0,1,0], [2,1,1], [2,10,1], [3,5,1], [3,6,1], [4,5,1], [4,6,1] };
k:=FiniteField(11);
NoThreeCollinear(P,k);
NoSixCoconic(P,k);
NoEightCosingularcubic(P,k); */


