/* This magma code accompanies the article B. Banwait, F. Fité, D. Loughran - Del Pezzo surfaces over finite fields and their Frobenius traces.

Its purpose to compile and verify our claims regarding Table 7.1. If you run it, it will do this in less than a minute

*/

/* Returns true if the given orbit consists of skew curves. */
is_skew := function(O, M)
 for i in O do
  for j in O do
   if i ne j and M[i,j] ne 0 then
    return false;
   end if;
  end for;
 end for;
 return true;
end function;

/* Constructs the characteristic sequence given by the orbit O. 
This is the sequence of intersection numbers of the vertices in the graph with the vertex */
char_seq := function(e,O,M)
 o1:=O[1];
 out:=[];
 for g in e do
  if Order(g) eq Order(e) then
   g0:=g;
  end if;
 end for;
 for n in [0..#O-1] do
  o:=o1^(g0^n);
  if o ne o1 then
   out:=Append(out, M[o1,o]);
  end if;
 end for;
 return out;
end function;
   

/* Given a skew orbit, it contracts it and displays the new orbit type of the del Pezzo surface thus obtained. */
contract_orbit := function(skew_orbit,e, M)
 if is_skew(skew_orbit,M) then
  new_orbit := [];
   for O in Orbits(e) do
    for o in O do
     keep := true;
     for i in skew_orbit do
       if o eq i or M[i,o] ge 1 then
        keep := false; 
        /* Keep o iff it does not meet a curve in the skew orbit. */
       end if;
     end for;
     if keep then
      new_orbit := Append(new_orbit, o);
     end if;
    end for;
  end for;
 end if;
return new_orbit;
end function;

Orbit_type := function(e)
 T:=[];
 for O in Orbits(e) do
  T:=Append(T,#O);
 end for;
 return T;
end function;

/*This code calculates the first Galois cohomology of the Picard group using [Manin, Proposition 31.3]. It was kindly given to us by Jörg Jahnel */
H1Pic := function(gr, deg, M)
 /* Jede Brauergruppe wird hier eingetragen werden. */

 /* Die Divisoren ueber dem algebraischen Abschlusz, zum Beispiel
 Z^27 bei kubischen Flaechen. */
 Div_space := RSpace(Integers(), deg);
 Hom_alg := Hom(Div_space, Div_space);
 int_map := Hom_alg!M;
 /* Die Hauptdivisoren */
 S_0 := Kernel(int_map);

 /* Alles haengt nur von der kombinatorischen Struktur der
 Orbiten ab. */
 orb := Orbits(gr);

 /* Die invarianten Divisoren */
 Div_space_arith := RSpace(Integers(), #orb);

 /* Die Normabbildung */
 N := ZeroMatrix(Integers(), deg,#orb);  
 for j := 1 to #orb do
  akt := orb[j];
  for k := 1 to #akt do
   /* Die Gerade akt[k] liegt im Orbit akt mit der Nummer j. */
   N[akt[k]][j] := #gr/#akt;
  end for;
 end for;
 norm_map := Hom(Div_space, Div_space_arith)!N;
 NS := Image(norm_map);

 /* Das Bild der Hauptdivisoren unter N. */
 NS_0 := norm_map(S_0);

 /* Die Einbettung der ionvarianten Divisoren in den Modul aller
 Divisoren */
 O := ZeroMatrix(Integers(), #orb,deg);
 for j := 1 to #orb do
  akt := orb[j];
  for k := 1 to #akt do
   /* Die Gerade akt[k] liegt im Orbit akt mit der Nummer j. */
   O[j][akt[k]] := 1;
  end for;
 end for;
 orb_map := Hom(Div_space_arith, Div_space)!O;
 
 /* int_map \circ orb_map !!! */
 /* NS geschnitten mit S_0 */
 linke_seite := NS meet Kernel(orb_map * int_map);

 /* Die Brauergruppe nach [Manin, Proposition 31.3] */
 Br := linke_seite / NS_0;
 vec := Moduli(Br);  /* Die abelschen Invarianten */
 return vec;
end function;



/************************** Degree 3 *******************************
*                                                                  *
*******************************************************************/
/* We now set-up all the groups relevant to Table 7.1 */
/* Creates W(E_6) together with its action on the 27 lines of a cubic surface */
printf "W(E_6)\n";
R_e6 := RootDatum("E6");
Cox_e6 := CoxeterGroup(R_e6);
we6 := StandardActionGroup(Cox_e6);
deg:=27;

/*This constructs the irreducible representation given by the orthogonal completement of the anticanonical class in Pic S
We work over the large finite field of order 1801 which contains all relevant roots of unity*/
q:=1801;
R:=Representation(AbsolutelyIrreducibleModules(we6, FiniteField(q))[3]);


/* Runs through the conjugacy classes of subgroups and finds those which are cyclic */
list := SubgroupClasses(we6);
erg := [];
for rec in list do
 gr := rec`subgroup;
 if IsCyclic(gr) then
  erg:=Append(erg, gr);
 end if;
end for;

/* Now sorts the subgroups so that they have the order as given in Table 7.1*/
Sorted:=[25,24,11,23,21,3,22,18,16,10,9,19,17,2,20,5,8,13,14,15,4,12,7,1,6];
Oerg:=erg;

for s in [1..25] do
 erg[Sorted[s]]:=Oerg[s];
end for;


/* Pairs of numbers in {1,..,27}. */
subs2 := Subsets(Set(GSet(we6)), 2);
orb := Orbits(we6, GSet(we6, subs2));
/* #orb[1]; */
/* 135 */
/* #orb[2]; */
/* 216 */

/* The intersection matrix of the graph*/
M := ZeroMatrix(Integers(), 27,27);
for i := 1 to 27 do
 for j := 1 to 27 do
  if i eq j then M[i,j] := -1; end if;
  if {i,j} in orb[1] then M[i,j] := 1; end if;
 end for;
end for;

/*We are now ready to compile the information relevant to Table 7.1*/
Total := 0;

/* Run through the list of cyclic subgroups */
for e in erg do
 Total := Total +1;
 
 for g in e do
  if Order(g) eq Order(e) then
   g0:=g;
  end if;
 end for;

 printf "%o. Subgroup order: %o. Measure^(-1): %o\n", Total, #e, #we6/#Class(we6,g0);

 /* Output the orbits*/
 printf "Orbits: %o \n", Orbit_type(e);


 /* We now look for skew orbits in order to calculate the index*/

 /*All possible collections of at most 6 orbits*/
 S:={};
 for i in [1..6] do
  S:=S join Subsets(Set(Orbits(e)),i);
 end for;

 /*We first assume that the index is 0, then keep increasing it as we find skew orbits of larger length*/
 Index:=0;
 has_skew_orbit := false;

 /*For a collection s of at most 6 orbits*/
 for s in S do 
  O:={};
  for o in s do
   O:=O join Set(o);
  end for;
  L:=#O;

  if L le 6 and L gt Index and is_skew(O, M) then
   skew_orbit := O; 
   has_skew_orbit := true; 
   Index:=Max(L,Index);
  end if;
 end for;

 if has_skew_orbit then
  printf "Skew orbit: [ "; 
  for O in Orbits(e, GSet(e,Set(skew_orbit))) do
   printf "%o ", #O;
  end for;
  printf "]. ";
  
  printf "New orbit type: [ ";
  new_orbit:=contract_orbit(skew_orbit,e,M);
  if not IsEmpty(new_orbit) then
   for O in Orbits(e, GSet(e,Set(new_orbit))) do
    printf "%o ", #O;
   end for;
  end if;
  printf "]. Index =  %o. \n", #skew_orbit;
 /*Else the surface is minimal*/
 else
  print "Index = 0";
 end if;
    
 /* Output H^1(Pic)*/
 printf "H^1(Pic) = %o \n", H1Pic(e,27,M);

 /*We now find the eigenvalues of the Frobenius acting on the Picard group,
 using the corresponding irreducible representation from above over FintieField(1801)*/
 eigen:=[1];
 for I in Eigenvalues(R(g0)) do
  for i in [1..I[2]] do
   eigen:=Append(eigen, Order(I[1]));
  end for;
 end for;
 a:=Integers()!(1+Trace(R(g0)));
 if a gt 7 then
  a:=a - q;
 end if;
 printf "Eigenvalues = %o \n", Sort(eigen);
 printf "Trace = %o \n", a;

 /* Output the characteristic sequence of orbits of length at least 2*/
 print "Characteristic sequences (for each non-trivial orbit, we output the intersections of each of the vertices with a given vertex):";
 for O in Orbits(e) do
  if #O ne 1 then
   seq:=char_seq(e,O,M);
   sum_seq:=0;
   for s in seq do
    sum_seq:=sum_seq+s;
   end for;
   printf "%o %o, \n", seq, sum_seq;
  end if;
 end for;
 print " ";
end for;



